/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *     Guido Tack <tack@gecode.org>
 *
 *  Copyright:
 *     Guido Tack, 2004
 *     Christian Schulte, 2005
 *
 *  Last modified:
 *     $Date: 2008-12-05 16:43:57 +0100 (Fri, 05 Dec 2008) $ by $Author: schulte $
 *     $Revision: 7989 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

namespace Gecode { namespace Iter { namespace Ranges { namespace Virt {

  /**
   * \brief Range iterator for computing the complement (described by template arguments)
   *
   * The complement is computed with respect to a given universe set
   * provided as template arguments (\a UMIN ... \a UMAX). The universe
   * must always be a superset!
   *
   * \ingroup FuncIterRangesVirt
   */

  template <int UMIN, int UMAX>
  class Compl : public MinMax, public Iterator {
  protected:
    /// Iterator to compute complement for
    Iterator* i;
    /// Initialize
    void start(void);
  public:
    /// \name Constructors and initialization
    //@{
    /// Default constructor
    Compl(void);
    /// Initialize with iterator \a i
    Compl(Iterator* i);
    /// Destructor
    ~Compl(void);
    //@}

    /// \name Range access
    //@{
    /// Return smallest value of range
    virtual int min(void) const;
    /// Return largest value of range
    virtual int max(void) const;
    /// Return width of range (distance between minimum and maximum)
    virtual unsigned int width(void) const;
    //@}

    /// \name Iteration control
    //@{
    /// Move iterator to next range (if possible)
    void operator ++(void);
    /// Test whether iterator is still at a range or done
    virtual bool operator ()(void);
    //@}
  };


  /**
   * \brief Range iterator for computing the complement (described by values)
   *
   * The complement is computed with respect to a given universe set
   * provided as arguments upon initialization. The universe
   * must always be a superset!
   *
   * \ingroup FuncIterRangesVirt
   */

  class ComplVal : public MinMax, public Iterator {
  protected:
    /// Values describing the universe set
    int UMIN, UMAX;
    /// Iterator to compute complement for
    Iterator* i;
    /// Initialize
    void start(void);
  public:
    /// \name Constructors and initialization
    //@{
    /// Default constructor
    ComplVal(void);
    /// Initialize with iterator \a i
    ComplVal(int umin, int umax, Iterator* i);
    /// Destructor
    ~ComplVal(void);
    //@}

    /// \name Range access
    //@{
    /// Return smallest value of range
    virtual int min(void) const;
    /// Return largest value of range
    virtual int max(void) const;
    /// Return width of range (distance between minimum and maximum)
    virtual unsigned int width(void) const;
    //@}

    /// \name Iteration control
    //@{
    /// Move iterator to next range (if possible)
    void operator ++(void);
    /// Test whether iterator is still at a range or done
    virtual bool operator ()(void);
    //@}
  };


  template <int UMIN, int UMAX>
  forceinline void
  Compl<UMIN,UMAX>::start(void) {
    if ((*i)()) {
      assert((i->min() >= UMIN) && (i->max() <= UMAX));
      if (i->min() > UMIN) {
        mi = UMIN;
        ma = i->min()-1;
      } else if (i->max() < UMAX) {
        mi = i->max()+1;
        ++(*i);
        ma = (*i)() ? (i->min()-1) : UMAX;
      } else {
        finish();
      }
    } else {
      mi = UMIN;
      ma = UMAX;
    }
  }

  template <int UMIN, int UMAX>
  forceinline
  Compl<UMIN,UMAX>::Compl(void) {}

  template <int UMIN, int UMAX>
  forceinline
  Compl<UMIN,UMAX>::Compl(Iterator* i0) : i(i0) {
    start();
  }

  template <int UMIN, int UMAX>
  Compl<UMIN,UMAX>::~Compl(void) { delete i; }

  template <int UMIN, int UMAX>
  forceinline void
  Compl<UMIN,UMAX>::operator ++(void) {
    assert(!(*i)() || (i->max() <= UMAX));
    if ((*i)() && (i->max() < UMAX)) {
      mi = i->max()+1;
      ++(*i);
      ma = (*i)() ? (i->min()-1) : UMAX;
    } else {
      finish();
    }
  }

  template <int UMIN, int UMAX>
  forceinline bool
  Compl<UMIN,UMAX>::operator ()(void) { return MinMax::operator ()(); }

  template <int UMIN, int UMAX>
  forceinline int
  Compl<UMIN,UMAX>::min(void) const { return MinMax::min(); }

  template <int UMIN, int UMAX>
  forceinline int
  Compl<UMIN,UMAX>::max(void) const { return MinMax::max(); }

  template <int UMIN, int UMAX>
  forceinline unsigned int
  Compl<UMIN,UMAX>::width(void) const { return MinMax::width(); }

  forceinline void
  ComplVal::start(void) {
    if ((*i)()) {
      assert((i->min() >= UMIN) && (i->max() <= UMAX));
      if (i->min() > UMIN) {
        mi = UMIN;
        ma = i->min()-1;
      } else if (i->max() < UMAX) {
        mi = i->max()+1;
        ++(*i);
        ma = (*i)() ? (i->min()-1) : UMAX;
      } else {
        finish();
      }
    } else {
      mi = UMIN;
      ma = UMAX;
    }
  }

  forceinline
  ComplVal::ComplVal(void) {}

  forceinline
  ComplVal::ComplVal(int umin, int umax, Iterator* i0)
    : UMIN(umin), UMAX(umax), i(i0) {
    start();
  }

  forceinline
  ComplVal::~ComplVal(void) { delete i; }

  forceinline void
  ComplVal::operator ++(void) {
    assert(!(*i)() || (i->max() <= UMAX));
    if ((*i)() && (i->max() < UMAX)) {
      mi = i->max()+1;
      ++(*i);
      ma = (*i)() ? (i->min()-1) : UMAX;
    } else {
      finish();
    }
  }

  forceinline bool
  ComplVal::operator ()(void) { return MinMax::operator ()(); }

  forceinline int
  ComplVal::min(void) const { return MinMax::min(); }

  forceinline int
  ComplVal::max(void) const { return MinMax::max(); }

  forceinline unsigned int
  ComplVal::width(void) const { return MinMax::width(); }

}}}}

// STATISTICS: iter-any

